//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES.
//
//===----------------------------------------------------------------------===//

// Modifications Copyright (c) 2025 Advanced Micro Devices, Inc.
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

// <memory>

// template <class T, class Alloc> struct uses_allocator;

#include <cuda/std/__memory_>
#if defined(_LIBCUDACXX_HAS_VECTOR)
#  include <cuda/std/vector>
#endif // _LIBCUDACXX_HAS_VECTOR
#include <cuda/std/type_traits>

#include "test_macros.h"

struct A
{};

struct B
{
  typedef int allocator_type;
};

#if !defined(TEST_COMPILER_NVRTC) && !defined(TEST_COMPILER_HIPRTC)
struct C
{
  static int allocator_type;
};
#endif // !TEST_COMPILER_NVRTC

struct D
{
  __host__ __device__ static int allocator_type()
  {
    return 0;
  }
};

struct E
{
private:
  typedef int allocator_type;
};

template <bool Expected, class T, class A>
__host__ __device__ void test()
{
  static_assert((cuda::std::uses_allocator<T, A>::value == Expected), "");
  static_assert(
    cuda::std::is_base_of<cuda::std::integral_constant<bool, Expected>, cuda::std::uses_allocator<T, A>>::value, "");
#if TEST_STD_VER >= 2017
  ASSERT_SAME_TYPE(decltype(cuda::std::uses_allocator_v<T, A>), const bool);
  static_assert((cuda::std::uses_allocator_v<T, A> == Expected), "");
#endif // TEST_STD_VER >= 2017
}

int main(int, char**)
{
  test<false, int, cuda::std::allocator<int>>();
#if defined(_LIBCUDACXX_HAS_VECTOR)
  test<true, cuda::std::vector<int>, cuda::std::allocator<int>>();
#endif //_LIBCUDACXX_HAS_VECTOR
  test<false, A, cuda::std::allocator<int>>();
  test<false, B, cuda::std::allocator<int>>();
  test<true, B, double>();
#if !defined(TEST_COMPILER_NVRTC) && !defined(TEST_COMPILER_HIPRTC)
  test<false, C, decltype(C::allocator_type)>();
#endif // !TEST_COMPILER_NVRTC
  test<false, D, decltype(D::allocator_type)>();
#if !defined(TEST_COMPILER_GCC) && !defined(TEST_COMPILER_MSVC_2017) // E::allocator_type is private
  test<false, E, int>();
#endif // !TEST_COMPILER_GCC && !TEST_COMPILER_MSVC_2017

  static_assert((!cuda::std::uses_allocator<int, cuda::std::allocator<int>>::value), "");
#if defined(_LIBCUDACXX_HAS_VECTOR)
  static_assert((cuda::std::uses_allocator<cuda::std::vector<int>, cuda::std::allocator<int>>::value), "");
#endif // _LIBCUDACXX_HAS_VECTOR
  static_assert((!cuda::std::uses_allocator<A, cuda::std::allocator<int>>::value), "");
  static_assert((!cuda::std::uses_allocator<B, cuda::std::allocator<int>>::value), "");
  static_assert((cuda::std::uses_allocator<B, double>::value), "");
#if !defined(TEST_COMPILER_NVRTC) && !defined(TEST_COMPILER_HIPRTC)
  static_assert((!cuda::std::uses_allocator<C, decltype(C::allocator_type)>::value), "");
  static_assert((!cuda::std::uses_allocator<D, decltype(D::allocator_type)>::value), "");
#endif // !TEST_COMPILER_NVRTC
#if !defined(TEST_COMPILER_GCC) && !defined(TEST_COMPILER_MSVC_2017) // E::allocator_type is private
  static_assert((!cuda::std::uses_allocator<E, int>::value), "");
#endif // !TEST_COMPILER_GCC && !TEST_COMPILER_MSVC_2017

  return 0;
}
